<?php
/**
 * Plugin Name: Easy ALT Import PRO
 * Description: Version PRO - Mise à jour illimitée des balises ALT, descriptions et captions d'images via CSV avec interface de prévisualisation avancée.
 * Version: 1.6 PRO
 * Author: Ftmpub.ch
 * Author URI: https://ftmpub.ch
 * Author Email: info@ftmpub.ch
 * Text Domain: easy-alt-import-pro
 * Domain Path: /languages
 * License: Commercial License
 * License URI: https://ftmpub.ch/license
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants - VERSION PRO
define('EASY_ALT_IMPORT_VERSION', '1.6 PRO');
define('EASY_ALT_IMPORT_IS_PRO_VERSION', true);

// Create admin menu
add_action('admin_menu', function () {
    add_menu_page(
        __('Easy ALT DESC CAPT Import PRO', 'easy-alt-import-pro'), 
        __('Easy ALT Import PRO', 'easy-alt-import-pro'), 
        'manage_options', 
        'easy-alt-import-pro', 
        'easy_alt_import_page', 
        'dashicons-upload', 
        80
    );
});

// PRO VERSION - Always return true
function easy_alt_import_is_pro() {
    return true;
}

// PRO VERSION - No monthly limits, but keep stats
function easy_alt_import_get_monthly_count() {
    $stats = get_option('easy_alt_import_pro_total_stats', ['total_processed' => 0]);
    return intval($stats['total_processed']);
}

// PRO VERSION - Update total stats instead of monthly limits
function easy_alt_import_update_monthly_count($count) {
    $stats = get_option('easy_alt_import_pro_total_stats', [
        'total_processed' => 0,
        'total_imports' => 0,
        'last_import' => null
    ]);
    
    $stats['total_processed'] += intval($count);
    $stats['total_imports']++;
    $stats['last_import'] = current_time('mysql');
    
    update_option('easy_alt_import_pro_total_stats', $stats);
}

// PRO VERSION - Always allow import (unlimited)
function easy_alt_import_can_import($requested_count = 0) {
    return [
        'can_import' => true,
        'remaining' => 'unlimited',
        'used' => 0,
        'limit' => 'unlimited'
    ];
}

// Secure CSV Upload & Processing - PRO VERSION
add_action('admin_post_easy_alt_import_pro_upload', function () {
    if (!current_user_can('manage_options') || !check_admin_referer('easy_alt_import_pro')) {
        wp_die(esc_html__('Action non autorisée', 'easy-alt-import-pro'));
    }

    if (empty($_FILES['csv_file']['tmp_name'])) {
        wp_redirect(admin_url('admin.php?page=easy-alt-import-pro&error=no_file'));
        exit;
    }

    // Vérifier les erreurs d'upload
    if (isset($_FILES['csv_file']['error']) && $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        wp_die(esc_html__('Erreur lors de l\'upload du fichier.', 'easy-alt-import-pro'));
    }

    // Vérifier l'extension du fichier
    $file_extension = '';
    if (isset($_FILES['csv_file']['name'])) {
        $file_extension = strtolower(pathinfo(sanitize_file_name($_FILES['csv_file']['name']), PATHINFO_EXTENSION));
    }
    
    if (!in_array($file_extension, ['csv', 'txt'])) {
        wp_die(esc_html__('Seuls les fichiers .csv et .txt sont autorisés.', 'easy-alt-import-pro'));
    }

    try {
        // Vérifier le type MIME - PRO: plus de formats acceptés
        $tmp_name = isset($_FILES['csv_file']['tmp_name']) ? sanitize_text_field($_FILES['csv_file']['tmp_name']) : '';
        if (empty($tmp_name)) {
            wp_die(esc_html__('Fichier temporaire non trouvé.', 'easy-alt-import-pro'));
        }
        
        $file_info = finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = finfo_file($file_info, $tmp_name);
        finfo_close($file_info);

        if (!in_array($mime_type, ['text/csv', 'text/plain', 'application/csv', 'application/vnd.ms-excel'])) {
            wp_die(esc_html__('Le fichier doit être au format CSV.', 'easy-alt-import-pro'));
        }

        // PRO VERSION - Limite de fichier augmentée à 20MB
        if (isset($_FILES['csv_file']['size']) && $_FILES['csv_file']['size'] > 20 * 1024 * 1024) {
            wp_die(esc_html__('Le fichier CSV ne peut pas dépasser 20MB.', 'easy-alt-import-pro'));
        }

        // Lire le fichier avec gestion de l'encodage améliorée
        $csv_content = file_get_contents($tmp_name);

        // PRO: Meilleure détection d'encodage
        if (!mb_check_encoding($csv_content, 'UTF-8')) {
            $detected_encoding = mb_detect_encoding($csv_content, ['UTF-8', 'ISO-8859-1', 'Windows-1252', 'ASCII'], true);
            if ($detected_encoding) {
                $csv_content = mb_convert_encoding($csv_content, 'UTF-8', $detected_encoding);
            } else {
                $csv_content = mb_convert_encoding($csv_content, 'UTF-8', 'auto');
            }
        }

        $csv_raw = str_getcsv($csv_content, "\n");
        if (empty($csv_raw)) {
            wp_die(esc_html__('Le fichier CSV semble vide.', 'easy-alt-import-pro'));
        }

        // ✅ AUTO-DÉTECTION INTELLIGENTE DU SÉPARATEUR - PRO
        $first_line = $csv_raw[0] ?? '';
        $semicolon_count = substr_count($first_line, ';');
        $comma_count = substr_count($first_line, ',');
        $tab_count = substr_count($first_line, "\t");

        // Déterminer le meilleur séparateur
        if ($semicolon_count > $comma_count && $semicolon_count > $tab_count) {
            $separator = ';';
        } elseif ($tab_count > $comma_count && $tab_count > $semicolon_count) {
            $separator = "\t";
        } else {
            $separator = ','; // Par défaut
        }

        // Parser avec le bon séparateur
        $csv = array_map(function($line) use ($separator) {
            return str_getcsv($line, $separator);
        }, $csv_raw);

        // Vérification de la qualité du parsing avec fallback intelligent
        $expected_columns = 4;
        $valid_rows = 0;
        foreach ($csv as $row) {
            if (count($row) === $expected_columns) {
                $valid_rows++;
            }
        }

        // Si moins de 80% des lignes sont correctes, essayer un autre séparateur
        if (count($csv) > 1 && ($valid_rows / count($csv)) < 0.8) {
            // Fallback : essayer tous les séparateurs
            $separators = [',', ';', "\t"];
            $best_separator = ',';
            $best_score = 0;
            
            foreach ($separators as $test_sep) {
                $test_csv = array_map(function($line) use ($test_sep) {
                    return str_getcsv($line, $test_sep);
                }, $csv_raw);
                
                $test_valid = 0;
                foreach ($test_csv as $row) {
                    if (count($row) === $expected_columns) {
                        $test_valid++;
                    }
                }
                
                $score = count($test_csv) > 0 ? ($test_valid / count($test_csv)) : 0;
                if ($score > $best_score) {
                    $best_score = $score;
                    $best_separator = $test_sep;
                }
            }
            
            // Re-parser avec le meilleur séparateur
            if ($best_separator !== $separator) {
                $csv = array_map(function($line) use ($best_separator) {
                    return str_getcsv($line, $best_separator);
                }, $csv_raw);
            }
        }

        if (count($csv) < 2) {
            wp_die(esc_html__('Le fichier CSV doit contenir au moins une ligne de données.', 'easy-alt-import-pro'));
        }

        $raw_header = array_shift($csv);

        // Supprimer le BOM s'il est présent
        if (isset($raw_header[0])) {
            $raw_header[0] = preg_replace('/^\xEF\xBB\xBF/', '', $raw_header[0]);
        }

        $headers = array_map('trim', $raw_header);
        $expected = ['id', 'alt', 'description', 'caption'];
        $headers_clean = array_map('strtolower', $headers);

        if (array_diff($expected, $headers_clean)) {
            wp_die(esc_html__('Erreur : votre fichier CSV doit contenir exactement les colonnes suivantes : ID, ALT, DESCRIPTION, CAPTION', 'easy-alt-import-pro'));
        }

        $rows = [];
        $errors = [];

        foreach ($csv as $line_num => $line) {
            if (count($line) !== count($headers)) {
                // translators: %d is the line number
                $errors[] = sprintf(__('Ligne %1$d : nombre de colonnes incorrect', 'easy-alt-import-pro'), $line_num + 2);
                continue;
            }

            $item = array_combine($headers, array_map('trim', $line));
            $id = intval($item['ID']);

            if ($id <= 0) {
                // translators: %1$d is the line number, %2$s is the invalid ID
                $errors[] = sprintf(__('Ligne %1$d : ID invalide (%2$s)', 'easy-alt-import-pro'), $line_num + 2, sanitize_text_field($item['ID']));
                continue;
            }

            if (get_post_type($id) !== 'attachment') {
                // translators: %1$d is the line number, %2$d is the ID
                $errors[] = sprintf(__('Ligne %1$d : ID %2$d n\'est pas un attachment', 'easy-alt-import-pro'), $line_num + 2, $id);
                continue;
            }

            // Vérifier si l'attachment est une image
            if (!wp_attachment_is_image($id)) {
                // translators: %1$d is the line number, %2$d is the ID
                $errors[] = sprintf(__('Ligne %1$d : ID %2$d n\'est pas une image', 'easy-alt-import-pro'), $line_num + 2, $id);
                continue;
            }

            $rows[] = [
                'ID' => $id,
                'ALT' => sanitize_text_field($item['ALT']),
                'DESCRIPTION' => sanitize_textarea_field($item['DESCRIPTION']),
                'CAPTION' => sanitize_textarea_field($item['CAPTION']),
                'current_alt' => get_post_meta($id, '_wp_attachment_image_alt', true),
                'current_description' => get_post_field('post_content', $id),
                'current_caption' => get_post_field('post_excerpt', $id)
            ];
        }

        // PRO VERSION - Pas de vérification de limites !

        update_option('easy_alt_import_pro_errors', $errors);
        update_option('easy_alt_import_pro_last_rows', $rows);

        $redirect_url = admin_url('admin.php?page=easy-alt-import-pro&preview=1');
        if (!empty($errors)) {
            $redirect_url .= '&errors=1';
        }

        wp_redirect($redirect_url);
        exit;

    } catch (Exception $e) {
        wp_die(esc_html__('Erreur lors du traitement du fichier CSV : ', 'easy-alt-import-pro') . esc_html($e->getMessage()));
    }
});

/* ========================================
   FIN DE LA PARTIE 1/4 - FONCTIONS CORE
   ======================================== */
   
   /* ========================================
   PARTIE 2/4 - TEMPLATES CSV ET ACTIONS ADMIN
   ======================================== */

// Download CSV template - PRO VERSION AMÉLIORÉE
add_action('admin_post_easy_alt_import_pro_download_template', function () {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Non autorisé', 'easy-alt-import-pro'));
    }
    
    // Validation avec nonce pour sécurité
    if (isset($_GET['_wpnonce']) && !wp_verify_nonce(sanitize_text_field($_GET['_wpnonce']), 'download_template')) {
        wp_die(esc_html__('Action non autorisée - nonce invalide', 'easy-alt-import-pro'));
    }
    
    // Récupérer le format demandé (international ou français)
    $format = 'intl';
    if (isset($_GET['format'])) {
        $format = sanitize_text_field(wp_unslash($_GET['format']));
    }
    
    $separator = ($format === 'fr') ? ';' : ',';
    $filename_suffix = ($format === 'fr') ? '-francais' : '-international';
    
    // PRO: Plus d'images dans le template
    $attachments = get_posts([
        'post_type' => 'attachment',
        'post_mime_type' => 'image',
        'numberposts' => 10, // Plus d'exemples en PRO
        'post_status' => 'inherit',
        'meta_query' => [
            [
                'key' => '_wp_attached_file',
                'compare' => 'EXISTS'
            ]
        ]
    ]);
    
    // Générer le contenu CSV avec le bon séparateur
    $csv_content = "ID{$separator}ALT{$separator}DESCRIPTION{$separator}CAPTION\n";
    
    if (!empty($attachments)) {
        foreach ($attachments as $attachment) {
            $current_alt = get_post_meta($attachment->ID, '_wp_attachment_image_alt', true);
            $current_description = get_post_field('post_content', $attachment->ID);
            $current_caption = get_post_field('post_excerpt', $attachment->ID);
            
            $csv_content .= sprintf(
                "%d{$separator}%s{$separator}%s{$separator}%s\n",
                $attachment->ID,
                $current_alt ?: __('Votre texte ALT ici', 'easy-alt-import-pro'),
                $current_description ?: __('Votre description ici', 'easy-alt-import-pro'),
                $current_caption ?: __('Votre légende ici', 'easy-alt-import-pro')
            );
        }
    } else {
        // Si pas d'images, créer des exemples améliorés PRO
        $csv_content .= "123{$separator}" . __('Image produit - Smartphone noir élégant', 'easy-alt-import-pro') . "{$separator}" . __('Smartphone dernière génération avec écran OLED 6.5 pouces, design premium en aluminium noir mat', 'easy-alt-import-pro') . "{$separator}" . __('Smartphone Pro Max - Innovation', 'easy-alt-import-pro') . "\n";
        $csv_content .= "124{$separator}" . __('Photo portrait professionnel femme souriante', 'easy-alt-import-pro') . "{$separator}" . __('Portrait professionnel d\'une femme d\'affaires souriante en costume bleu marine sur fond blanc', 'easy-alt-import-pro') . "{$separator}" . __('CEO Sarah Martin - Leadership', 'easy-alt-import-pro') . "\n";
        $csv_content .= "125{$separator}" . __('Paysage montagne coucher soleil', 'easy-alt-import-pro') . "{$separator}" . __('Magnifique paysage de montagne au coucher du soleil avec ciel orange et violet, reflets dorés sur les sommets enneigés', 'easy-alt-import-pro') . "{$separator}" . __('Alpes françaises - Nature majestueuse', 'easy-alt-import-pro') . "\n";
    }
    
    // Headers pour forcer le téléchargement
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="easy-alt-import-template' . sanitize_file_name($filename_suffix) . '-' . gmdate('Y-m-d') . '.csv"');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    // Ajouter BOM UTF-8 pour Excel et échapper le contenu
    echo "\xEF\xBB\xBF" . wp_kses_post($csv_content);
    exit;
});

// Undo last import - PRO VERSION
add_action('admin_post_easy_alt_import_pro_undo', function () {
    if (!current_user_can('manage_options') || !check_admin_referer('easy_alt_import_pro_undo')) {
        wp_die(esc_html__('Action non autorisée', 'easy-alt-import-pro'));
    }

    $backup = get_option('easy_alt_import_pro_backup');
    $processed = 0;

    if ($backup && is_array($backup)) {
        foreach ($backup as $id => $data) {
            $id = intval($id); // Sécuriser l'ID
            
            try {
                if (get_post_type($id) === 'attachment') {
                    // Restore ALT text
                    if (isset($data['alt'])) {
                        update_post_meta($id, '_wp_attachment_image_alt', sanitize_text_field($data['alt']));
                    }
                    
                    // Restore description and caption
                    $update_data = ['ID' => $id];
                    if (isset($data['post_content'])) {
                        $update_data['post_content'] = wp_kses_post($data['post_content']);
                    }
                    if (isset($data['post_excerpt'])) {
                        $update_data['post_excerpt'] = sanitize_text_field($data['post_excerpt']);
                    }
                    
                    wp_update_post($update_data);
                    $processed++;
                }
            } catch (Exception $e) {
                // Pas d'error_log en production - on ignore silencieusement
                continue;
            }
        }
    }

    delete_option('easy_alt_import_pro_backup');
    delete_option('easy_alt_import_pro_last_rows');
    delete_option('easy_alt_import_pro_errors');

    wp_redirect(admin_url('admin.php?page=easy-alt-import-pro&undo=1&processed=' . intval($processed)));
    exit;
});

// Apply import - PRO VERSION avec sécurité renforcée
add_action('admin_post_easy_alt_import_pro_apply', function () {
    if (!current_user_can('manage_options') || !check_admin_referer('easy_alt_import_pro')) {
        wp_die(esc_html__('Non autorisé', 'easy-alt-import-pro'));
    }

    $rows = get_option('easy_alt_import_pro_last_rows', []);
    $selected_ids = [];
    
    if (isset($_POST['selected_ids']) && is_array($_POST['selected_ids'])) {
        $selected_ids = array_map('intval', $_POST['selected_ids']);
    }
    
    // Filtrer les lignes sélectionnées uniquement
    $filtered_rows = array_filter($rows, function($row) use ($selected_ids) {
        return in_array(intval($row['ID']), $selected_ids);
    });
    
    // PRO VERSION - Pas de vérification de limites !

    $backup = [];
    $processed = 0;
    $errors = [];

    foreach ($filtered_rows as $r) {
        $id = intval($r['ID']);

        try {
            // Créer une sauvegarde avant modification
            $backup[$id] = [
                'alt' => get_post_meta($id, '_wp_attachment_image_alt', true),
                'post_content' => get_post_field('post_content', $id),
                'post_excerpt' => get_post_field('post_excerpt', $id),
            ];

            // Appliquer les modifications
            update_post_meta($id, '_wp_attachment_image_alt', sanitize_text_field($r['ALT']));
            wp_update_post([
                'ID' => $id,
                'post_content' => wp_kses_post($r['DESCRIPTION']),
                'post_excerpt' => sanitize_text_field($r['CAPTION'])
            ]);

            $processed++;

        } catch (Exception $e) {
            // translators: %1$d is the image ID, %2$s is the error message
            $errors[] = sprintf(__('Erreur lors de la mise à jour de l\'ID %1$d : %2$s', 'easy-alt-import-pro'), $id, esc_html($e->getMessage()));
        }
    }

    // Sauvegarder les données de rollback
    update_option('easy_alt_import_pro_backup', $backup);
    update_option('easy_alt_import_pro_backup_date', current_time('mysql'));
    
    // Mettre à jour les statistiques PRO
    easy_alt_import_update_monthly_count($processed);

    // Nettoyer les données temporaires
    delete_option('easy_alt_import_pro_last_rows');
    delete_option('easy_alt_import_pro_errors');

    $redirect_url = admin_url('admin.php?page=easy-alt-import-pro&done=1&processed=' . intval($processed));
    if (!empty($errors)) {
        update_option('easy_alt_import_pro_apply_errors', $errors);
        $redirect_url .= '&apply_errors=1';
    }

    wp_redirect($redirect_url);
    exit;
});

// Add admin notice for successful activation
add_action('admin_notices', function() {
    if (get_transient('easy_alt_import_pro_activated')) {
        echo '<div class="notice notice-success is-dismissible">';
        echo '<p><strong>🎉 ' . esc_html__('Easy ALT Import PRO activé avec succès !', 'easy-alt-import-pro') . '</strong> ';
        echo '<a href="' . esc_url(admin_url('admin.php?page=easy-alt-import-pro')) . '">' . esc_html__('Commencer l\'optimisation SEO →', 'easy-alt-import-pro') . '</a></p>';
        echo '</div>';
        delete_transient('easy_alt_import_pro_activated');
    }
});

// Cleanup old data on plugin deactivation
register_deactivation_hook(__FILE__, function() {
    // Clean up temporary options, but keep PRO stats for user retention
    delete_option('easy_alt_import_pro_last_rows');
    delete_option('easy_alt_import_pro_errors');
    delete_option('easy_alt_import_pro_apply_errors');
    // Note: We keep 'easy_alt_import_pro_total_stats' and 'easy_alt_import_pro_backup' for reactivation
});

/* ========================================
   FIN DE LA PARTIE 2/4 - TEMPLATES & ACTIONS
   ======================================== */
   
    /* ========================================
   PARTIE 3/4 - INTERFACE UTILISATEUR (DÉBUT)
   ======================================== */

// Main admin page - PRO VERSION COMPLÈTE AMÉLIORÉE
function easy_alt_import_page() {
    $import_limits = easy_alt_import_can_import();
    $pro_stats = get_option('easy_alt_import_pro_total_stats', ['total_processed' => 0, 'total_imports' => 0]);
    
    echo '<div class="wrap">';
    echo '<h1 style="margin-bottom:1em;">' . esc_html__('Easy ALT DESC CAPT CSV Import', 'easy-alt-import-pro') . ' <span style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 5px 12px; border-radius: 15px; font-size: 0.6em; margin-left: 10px;">' . esc_html__('PRO', 'easy-alt-import-pro') . '</span></h1>';

    // CSS Styles - PRO VERSION avec thème premium amélioré
    echo '<style>
    .easy-alt-container { max-width: 1200px; }
    .easy-alt-card { background: #fff; border: 1px solid #ccd0d4; padding: 20px; margin: 20px 0; box-shadow: 0 1px 1px rgba(0,0,0,.04); border-radius: 8px; }
    .easy-alt-pro-banner { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; margin: 20px 0; text-align: center; }
    .easy-alt-limits { background: linear-gradient(135deg, #e8f5e8 0%, #f0f8f0 100%); border-left: 4px solid #46b450; padding: 20px; margin: 20px 0; border-radius: 5px; }
    .easy-alt-steps { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin: 20px 0; }
    .step { background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); padding: 15px; border-radius: 8px; text-align: center; border-left: 4px solid #667eea; font-weight: bold; }
    .help-section { background: #e7f3ff; border: 1px solid #b3d9ff; padding: 15px; margin: 15px 0; border-radius: 8px; }
    .csv-preview { background: #f8f9fa; border: 1px solid #dee2e6; padding: 10px; margin: 10px 0; border-radius: 5px; font-family: monospace; font-size: 0.9em; }
    table.widefat img { max-height: 60px; width: auto; }
    .easy-alt-footer { margin-top:2em; font-size:0.9em; color:#666; }
    .error-list { background: #ffeaea; border: 1px solid #dc3232; padding: 15px; margin: 15px 0; border-radius: 5px; }
    .error-list h3 { color: #dc3232; margin-top: 0; }
    .success-message { background: #e8f5e8; border: 1px solid #46b450; padding: 15px; margin: 15px 0; color: #46b450; border-radius: 5px; }
    .warning-message { background: #fff3cd; border: 1px solid #ffc107; padding: 15px; margin: 15px 0; color: #856404; border-radius: 5px; }
    .comparison-table { width: 100%; }
    .comparison-table th { background: #f1f1f1; }
    .comparison-table .current { background: #f9f9f9; }
    .comparison-table .new { background: #e8f5e8; }
    .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin: 20px 0; }
    .stat-card { background: #fff; border: 1px solid #ccd0d4; padding: 15px; text-align: center; border-radius: 5px; }
    .stat-card.pro { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; }
    .stat-number { font-size: 2em; font-weight: bold; color: #007cba; }
    .stat-card.pro .stat-number { color: white; }
    .selection-controls { margin: 15px 0; padding: 10px; background: #f8f9fa; border-radius: 5px; }
    .selection-controls button { margin-right: 10px; }
    .selected-count { font-weight: bold; color: #007cba; margin-left: 10px; }
    .pro-feature { background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); border-left: 4px solid #667eea; padding: 15px; margin: 15px 0; border-radius: 5px; }
    .media-library-link { background: linear-gradient(135deg, #0073aa 0%, #005177 100%); color: white; text-decoration: none; padding: 8px 15px; border-radius: 5px; display: inline-block; margin: 10px 5px; }
    .media-library-link:hover { color: white; text-decoration: none; }
    details { margin: 15px 0; }
    summary { cursor: pointer; font-weight: bold; padding: 10px; background: #f8f9fa; border-radius: 5px; }
    details[open] summary { border-bottom: 1px solid #dee2e6; border-radius: 5px 5px 0 0; }
    details[open] > *:not(summary) { padding: 10px; background: #f8f9fa; border-radius: 0 0 5px 5px; margin: 0; }
    .notice.notice-info { background: #e7f3ff; border-left: 4px solid #0073aa; }
    .notice.notice-success { background: #e8f5e8; border-left: 4px solid #46b450; }
    .notice.notice-warning { background: #fff3cd; border-left: 4px solid #ffc107; }
    </style>';

    echo '<div class="easy-alt-container">';

    // PRO Banner
    echo '<div class="easy-alt-pro-banner">';
    echo '<h2 style="margin: 0 0 10px 0;">' . esc_html__('Version PRO Activée !', 'easy-alt-import-pro') . '</h2>';
    echo '<p style="margin: 0; opacity: 0.9;">' . esc_html__('Imports illimités • Auto-détection CSV • Fichiers 20MB • Support prioritaire • Encodages avancés', 'easy-alt-import-pro') . '</p>';
    echo '</div>';

    // PRO Stats display avec encouragements
    echo '<div class="easy-alt-limits">';
    echo '<h3>' . esc_html__('Statistiques PRO', 'easy-alt-import-pro') . '</h3>';
    echo '<div class="stats-grid">';
    echo '<div class="stat-card pro"><div class="stat-number">∞</div><div>' . esc_html__('Imports illimités', 'easy-alt-import-pro') . '</div></div>';
    echo '<div class="stat-card pro"><div class="stat-number">' . esc_html(number_format($pro_stats['total_processed'])) . '</div><div>' . esc_html__('Images optimisées', 'easy-alt-import-pro') . '</div></div>';
    echo '<div class="stat-card pro"><div class="stat-number">' . esc_html(number_format($pro_stats['total_imports'])) . '</div><div>' . esc_html__('Imports réussis', 'easy-alt-import-pro') . '</div></div>';
    echo '<div class="stat-card pro"><div class="stat-number">20MB</div><div>' . esc_html__('Taille max fichier', 'easy-alt-import-pro') . '</div></div>';
    echo '</div>';
    
    // Message d'encouragement selon les stats
    if ($pro_stats['total_processed'] == 0) {
        echo '<div class="notice notice-info">';
        echo '<p>🚀 <strong>' . esc_html__('Prêt à optimiser votre SEO ?', 'easy-alt-import-pro') . '</strong> ' . esc_html__('Commencez par télécharger le template avec vos images existantes !', 'easy-alt-import-pro') . '</p>';
        echo '</div>';
    } elseif ($pro_stats['total_processed'] < 100) {
        echo '<div class="notice notice-success">';
        echo '<p>✨ <strong>' . esc_html__('Excellent début !', 'easy-alt-import-pro') . '</strong> ' . esc_html__('Vous optimisez déjà votre référencement naturel.', 'easy-alt-import-pro') . '</p>';
        echo '</div>';
    } else {
        echo '<div class="notice notice-success">';
        echo '<p>🏆 <strong>' . esc_html__('Expert SEO !', 'easy-alt-import-pro') . '</strong> ' . sprintf(esc_html__('Plus de %d images optimisées - Votre site gagne en visibilité !', 'easy-alt-import-pro'), $pro_stats['total_processed']) . '</p>';
        echo '</div>';
    }
    echo '</div>';

    // Messages d'état améliorés
    if (isset($_GET['done'])) {
        $processed = isset($_GET['processed']) ? intval($_GET['processed']) : 0;
        echo '<div class="success-message">';
        echo '<strong>🎉 ' . esc_html__('Importation PRO réussie !', 'easy-alt-import-pro') . '</strong> ' . sprintf(esc_html__('%d images ont été mises à jour. Votre SEO s\'améliore !', 'easy-alt-import-pro'), $processed);
        echo '</div>';
    }

    if (isset($_GET['undo'])) {
        $processed = isset($_GET['processed']) ? intval($_GET['processed']) : 0;
        echo '<div class="success-message">';
        echo '<strong>↩️ ' . esc_html__('Annulation réussie !', 'easy-alt-import-pro') . '</strong> ' . sprintf(esc_html__('%d modifications ont été annulées.', 'easy-alt-import-pro'), $processed);
        echo '</div>';
    }

    if (isset($_GET['error'])) {
        echo '<div class="error-list">';
        echo '<h3>' . esc_html__('Erreur', 'easy-alt-import-pro') . '</h3>';
        $error_type = isset($_GET['error']) ? sanitize_text_field(wp_unslash($_GET['error'])) : '';
        switch ($error_type) {
            case 'no_file':
                echo '<p>' . esc_html__('Aucun fichier sélectionné. Choisissez votre fichier CSV ci-dessous.', 'easy-alt-import-pro') . '</p>';
                break;
            default:
                echo '<p>' . esc_html__('Une erreur est survenue. Vérifiez le format de votre fichier CSV.', 'easy-alt-import-pro') . '</p>';
        }
        echo '</div>';
    }

    if (isset($_GET['apply_errors'])) {
        $errors = get_option('easy_alt_import_pro_apply_errors', []);
        if (!empty($errors)) {
            echo '<div class="error-list">';
            echo '<h3>' . esc_html__('Erreurs lors de l\'application', 'easy-alt-import-pro') . '</h3><ul>';
            foreach ($errors as $error) {
                echo '<li>' . esc_html($error) . '</li>';
            }
            echo '</ul></div>';
            delete_option('easy_alt_import_pro_apply_errors');
        }
    }

    // Rollback section améliorée
    if (!isset($_GET['preview'])) {
        $backup = get_option('easy_alt_import_pro_backup');
        if ($backup && is_array($backup)) {
            $backup_date = get_option('easy_alt_import_pro_backup_date');
            echo '<div class="easy-alt-card">';
            echo '<h3>↩️ ' . esc_html__('Annulation possible', 'easy-alt-import-pro') . '</h3>';
            echo '<p><strong>' . esc_html__('Dernière importation :', 'easy-alt-import-pro') . '</strong> ' . ($backup_date ? esc_html(gmdate('d/m/Y H:i', strtotime($backup_date))) : esc_html__('Date inconnue', 'easy-alt-import-pro')) . '</p>';
            echo '<p><strong>' . esc_html__('Images concernées :', 'easy-alt-import-pro') . '</strong> ' . esc_html(count($backup)) . '</p>';
            echo '<p class="description">' . esc_html__('Vous pouvez annuler la dernière importation pour revenir aux anciennes valeurs.', 'easy-alt-import-pro') . '</p>';
            echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" onsubmit="return confirm(\'' . 
                 esc_js(__('⚠️ Êtes-vous sûr de vouloir annuler la dernière importation ? Cette action est irréversible.', 'easy-alt-import-pro')) . '\');">';
            echo '<input type="hidden" name="action" value="easy_alt_import_pro_undo">';
            wp_nonce_field('easy_alt_import_pro_undo');
            submit_button(__('Annuler la dernière importation', 'easy-alt-import-pro'), 'delete', 'submit', false);
            echo '</form>';
            echo '</div>';
        }
    }

/* ========================================
   FIN DE LA PARTIE 3/4 - INTERFACE (DÉBUT)
   ======================================== */
   
   /* ========================================
   PARTIE 4/4 FINALE - INTERFACE UTILISATEUR (FIN)
   ======================================== */

    // Preview mode amélioré
    if (isset($_GET['preview'])) {
        $rows = get_option('easy_alt_import_pro_last_rows', []);
        $errors = get_option('easy_alt_import_pro_errors', []);

        // Amélioration des messages d'erreur
        if (!empty($errors)) {
            echo '<div class="error-list">';
            echo '<h3>⚠️ ' . esc_html__('Erreurs détectées dans votre CSV', 'easy-alt-import-pro') . '</h3>';
            echo '<p>' . esc_html__('Les lignes suivantes contiennent des erreurs et ne seront pas importées :', 'easy-alt-import-pro') . '</p>';
            echo '<ul>';
            foreach ($errors as $error) {
                echo '<li>' . esc_html($error) . '</li>';
            }
            echo '</ul>';
            
            // Suggestions d'amélioration
            if (count($errors) > count($rows)) {
                echo '<div class="help-section">';
                echo '<h4>💡 ' . esc_html__('Suggestions d\'amélioration :', 'easy-alt-import-pro') . '</h4>';
                echo '<ul>';
                echo '<li><a href="' . esc_url(admin_url('upload.php')) . '" target="_blank">📁 ' . 
                     esc_html__('Vérifiez vos IDs d\'images dans la médiathèque', 'easy-alt-import-pro') . '</a></li>';
                echo '<li>' . esc_html__('Assurez-vous que les IDs correspondent à de vraies images', 'easy-alt-import-pro') . '</li>';
                echo '<li>' . esc_html__('Vérifiez que votre CSV a bien 4 colonnes : ID, ALT, DESCRIPTION, CAPTION', 'easy-alt-import-pro') . '</li>';
                echo '</ul>';
                echo '</div>';
            }
            echo '</div>';
        }

        if (!empty($rows)) {
            echo '<div class="easy-alt-card">';
            echo '<h3>👀 ' . sprintf(esc_html__('Prévisualisation des données à importer (%d éléments)', 'easy-alt-import-pro'), esc_html(count($rows))) . '</h3>';
            
            // PRO VERSION - Toujours permettre l'import
            echo '<div class="pro-feature">';
            echo '<h4>🚀 ' . esc_html__('Mode PRO Actif', 'easy-alt-import-pro') . '</h4>';
            echo '<p>' . esc_html__('Tous les éléments peuvent être importés sans restriction ! Auto-détection CSV réussie.', 'easy-alt-import-pro') . '</p>';
            echo '</div>';
            
            echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" id="import-form">';
            echo '<input type="hidden" name="action" value="easy_alt_import_pro_apply">';
            wp_nonce_field('easy_alt_import_pro');
            
            echo '<div class="selection-controls">';
            echo '<button type="button" class="button" onclick="selectAll()">✅ ' . esc_html__('Tout sélectionner', 'easy-alt-import-pro') . '</button>';
            echo '<button type="button" class="button" onclick="deselectAll()">❌ ' . esc_html__('Tout désélectionner', 'easy-alt-import-pro') . '</button>';
            echo '<span class="selected-count">' . esc_html__('Sélectionnés:', 'easy-alt-import-pro') . ' <span id="selected-count">' . esc_html(count($rows)) . '</span>/' . esc_html(count($rows)) . '</span>';
            echo '</div>';
            
            echo '<div class="stats-grid">';
            echo '<div class="stat-card"><div class="stat-number">' . esc_html(count($rows)) . '</div><div>' . esc_html__('Images trouvées', 'easy-alt-import-pro') . '</div></div>';
            echo '<div class="stat-card pro"><div class="stat-number" id="selected-stat">' . esc_html(count($rows)) . '</div><div>' . esc_html__('Images sélectionnées', 'easy-alt-import-pro') . '</div></div>';
            echo '<div class="stat-card pro"><div class="stat-number">∞</div><div>' . esc_html__('Aucune limite', 'easy-alt-import-pro') . '</div></div>';
            echo '</div>';
            
            echo '<table class="widefat comparison-table">';
            echo '<thead><tr><th width="30">' . esc_html__('Sélection', 'easy-alt-import-pro') . '</th>';
            echo '<th>' . esc_html__('ID', 'easy-alt-import-pro') . '</th>';
            echo '<th>' . esc_html__('Image', 'easy-alt-import-pro') . '</th>';
            echo '<th>' . esc_html__('ALT Actuel → Nouveau', 'easy-alt-import-pro') . '</th>';
            echo '<th>' . esc_html__('Description', 'easy-alt-import-pro') . '</th>';
            echo '<th>' . esc_html__('Légende', 'easy-alt-import-pro') . '</th></tr></thead>';
            echo '<tbody>';
            
            foreach ($rows as $r) {
                echo '<tr>';
                echo '<td><input type="checkbox" name="selected_ids[]" value="' . esc_attr($r['ID']) . '" checked onchange="updateCount()"></td>';
                echo '<td>' . esc_html($r['ID']) . '</td>';
                echo '<td>' . wp_get_attachment_image($r['ID'], 'thumbnail') . '</td>';
                echo '<td>';
                echo '<div class="current"><strong>' . esc_html__('Actuel:', 'easy-alt-import-pro') . '</strong> ' . 
                     ($r['current_alt'] ? esc_html($r['current_alt']) : '<em>' . esc_html__('(vide)', 'easy-alt-import-pro') . '</em>') . '</div>';
                echo '<div class="new"><strong>' . esc_html__('Nouveau:', 'easy-alt-import-pro') . '</strong> ' . esc_html($r['ALT']) . '</div>';
                echo '</td>';
                echo '<td>' . esc_html(wp_trim_words($r['DESCRIPTION'], 10)) . '</td>';
                echo '<td>' . esc_html($r['CAPTION']) . '</td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
            
            submit_button('🚀 ' . __('Appliquer les modifications sélectionnées', 'easy-alt-import-pro'), 'primary', 'submit', false, [
                'style' => 'margin-top: 20px; font-size: 1.1em; padding: 12px 24px;',
                'onclick' => 'return confirmImport()'
            ]);
            echo '</form>';
            echo '</div>';
        } else {
            // Aucune donnée valide trouvée
            echo '<div class="error-list">';
            echo '<h3>❌ ' . esc_html__('Aucune image valide trouvée', 'easy-alt-import-pro') . '</h3>';
            echo '<p>' . esc_html__('Votre fichier CSV ne contient aucune image valide à importer.', 'easy-alt-import-pro') . '</p>';
            echo '<div class="help-section">';
            echo '<h4>🔧 ' . esc_html__('Solutions suggérées :', 'easy-alt-import-pro') . '</h4>';
            echo '<ul>';
            echo '<li><a href="' . esc_url(admin_url('upload.php')) . '" target="_blank">📁 ' . 
                 esc_html__('Uploadez d\'abord des images dans votre médiathèque', 'easy-alt-import-pro') . '</a></li>';
            echo '<li>' . esc_html__('Téléchargez le template ci-dessous avec vos vraies images', 'easy-alt-import-pro') . '</li>';
            echo '<li>' . esc_html__('Vérifiez que les IDs de votre CSV correspondent aux images existantes', 'easy-alt-import-pro') . '</li>';
            echo '</ul>';
            echo '</div>';
            echo '</div>';
        }
    } else {
        // Upload form - PRO VERSION AMÉLIORÉE
        echo '<div class="easy-alt-card">';
        echo '<h3>📤 ' . esc_html__('Importer un fichier CSV', 'easy-alt-import-pro') . ' <span style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 2px 8px; border-radius: 10px; font-size: 0.7em; margin-left: 5px;">' . esc_html__('PRO', 'easy-alt-import-pro') . '</span></h3>';
        
        // Étapes visuelles
        echo '<div class="easy-alt-steps">';
        echo '<div class="step">1️⃣ ' . esc_html__('Téléchargez le template', 'easy-alt-import-pro') . '</div>';
        echo '<div class="step">2️⃣ ' . esc_html__('Modifiez dans Excel', 'easy-alt-import-pro') . '</div>';
        echo '<div class="step">3️⃣ ' . esc_html__('Sauvegardez en CSV', 'easy-alt-import-pro') . '</div>';
        echo '<div class="step">4️⃣ ' . esc_html__('Uploadez ci-dessous', 'easy-alt-import-pro') . '</div>';
        echo '</div>';
        
        // Template download section améliorée
        echo '<div style="margin: 20px 0; padding: 15px; background: linear-gradient(135deg, #f0f8ff 0%, #e6f3ff 100%); border: 1px solid #b3d9ff; border-radius: 8px;">';
        echo '<h4 style="margin: 0 0 10px 0; color: #0066cc;">🚀 ' . esc_html__('Commencez rapidement !', 'easy-alt-import-pro') . '</h4>';
        echo '<p style="margin: 0 0 15px 0;">' . esc_html__('Téléchargez notre template intelligent pré-rempli avec vos images existantes :', 'easy-alt-import-pro') . '</p>';
        
        // Vérification des images disponibles
        $available_images = get_posts([
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'numberposts' => 1,
            'post_status' => 'inherit'
        ]);
        
        if (empty($available_images)) {
            echo '<div class="notice notice-warning">';
            echo '<p>⚠️ <strong>' . esc_html__('Aucune image trouvée !', 'easy-alt-import-pro') . '</strong> ';
            echo '<a href="' . esc_url(admin_url('upload.php')) . '" target="_blank" class="media-library-link">📁 ' . 
                 esc_html__('Uploadez d\'abord des images', 'easy-alt-import-pro') . '</a>';
            echo ' ' . esc_html__('puis revenez télécharger le template.', 'easy-alt-import-pro') . '</p>';
            echo '</div>';
        } else {
            echo '<div style="display: flex; gap: 10px; align-items: center; flex-wrap: wrap;">';
            echo '<a href="' . esc_url(wp_nonce_url(admin_url('admin-post.php?action=easy_alt_import_pro_download_template&format=intl'), 'download_template')) . '" class="button button-primary" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none; color: white; text-decoration: none;">📄 ' . 
                 esc_html__('Template International (,)', 'easy-alt-import-pro') . '</a>';
            echo '<a href="' . esc_url(wp_nonce_url(admin_url('admin-post.php?action=easy_alt_import_pro_download_template&format=fr'), 'download_template')) . '" class="button button-secondary" style="margin-left: 5px;">📄 ' . 
                 esc_html__('Template Français (;)', 'easy-alt-import-pro') . '</a>';
            echo '<a href="' . esc_url(admin_url('upload.php')) . '" target="_blank" class="media-library-link">📁 ' . 
                 esc_html__('Voir médiathèque', 'easy-alt-import-pro') . '</a>';
            echo '</div>';
        }
        
        echo '<p style="margin: 10px 0 0 0; font-size: 0.9em; color: #666;"><strong>💡 ' . esc_html__('Astuce PRO :', 'easy-alt-import-pro') . '</strong> ' . 
             esc_html__('Auto-détection des séparateurs - Aucun souci de format !', 'easy-alt-import-pro') . '</p>';
        echo '</div>';
        
        // Aide contextuelle améliorée
        echo '<details>';
        echo '<summary>💡 ' . esc_html__('Comment trouver l\'ID d\'une image ?', 'easy-alt-import-pro') . '</summary>';
        echo '<div style="margin-top: 10px;">';
        echo '<p>1. ' . esc_html__('Allez dans', 'easy-alt-import-pro') . ' <a href="' . esc_url(admin_url('upload.php')) . '" target="_blank">📁 Médias → Bibliothèque</a></p>';
        echo '<p>2. ' . esc_html__('Cliquez sur une image pour l\'ouvrir', 'easy-alt-import-pro') . '</p>';
        echo '<p>3. ' . esc_html__('L\'ID apparaît dans l\'URL :', 'easy-alt-import-pro') . ' <code>post.php?post=<strong>123</strong></code></p>';
        echo '<p>4. ' . esc_html__('Notez ces IDs pour votre CSV', 'easy-alt-import-pro') . '</p>';
        echo '</div>';
        echo '</details>';
        
        echo '<div class="pro-feature">';
        echo '<h4>🌟 ' . esc_html__('Fonctionnalités PRO', 'easy-alt-import-pro') . '</h4>';
        echo '<ul>';
        echo '<li><strong>✅ ' . esc_html__('Imports illimités', 'easy-alt-import-pro') . '</strong> - ' . esc_html__('Aucune restriction mensuelle', 'easy-alt-import-pro') . '</li>';
        echo '<li><strong>📁 ' . esc_html__('Fichiers jusqu\'à 20MB', 'easy-alt-import-pro') . '</strong> - ' . esc_html__('4x plus grand que la version gratuite', 'easy-alt-import-pro') . '</li>';
        echo '<li><strong>🧠 ' . esc_html__('Auto-détection CSV', 'easy-alt-import-pro') . '</strong> - ' . esc_html__('Virgules, point-virgules, tabulations', 'easy-alt-import-pro') . '</li>';
        echo '<li><strong>🌍 ' . esc_html__('Encodages multiples', 'easy-alt-import-pro') . '</strong> - ' . esc_html__('UTF-8, ISO-8859-1, Windows-1252', 'easy-alt-import-pro') . '</li>';
        echo '<li><strong>📊 ' . esc_html__('Template intelligent', 'easy-alt-import-pro') . '</strong> - ' . esc_html__('Pré-rempli avec vos vraies images', 'easy-alt-import-pro') . '</li>';
        echo '<li><strong>🔧 ' . esc_html__('Support prioritaire', 'easy-alt-import-pro') . '</strong> - ' . esc_html__('Assistance dédiée ftmpub.ch', 'easy-alt-import-pro') . '</li>';
        echo '</ul>';
        echo '</div>';
        
        echo '<form method="post" enctype="multipart/form-data" action="' . esc_url(admin_url('admin-post.php')) . '" style="max-width:600px; margin-top: 20px;">';
        echo '<input type="hidden" name="action" value="easy_alt_import_pro_upload">';
        wp_nonce_field('easy_alt_import_pro');
        echo '<div style="border: 2px dashed #667eea; padding: 20px; border-radius: 8px; text-align: center; margin: 20px 0; background: #f8f9ff;">';
        echo '<p style="margin: 0 0 15px 0; font-size: 1.1em;"><strong>📂 ' . esc_html__('Sélectionnez votre fichier CSV', 'easy-alt-import-pro') . '</strong></p>';
        echo '<input type="file" name="csv_file" accept=".csv,.txt" required style="margin-bottom:1em; width: 100%; padding: 10px; border: 1px solid #ccc; border-radius: 5px;" onchange="previewCSV(this)">';
        echo '<div id="csv-preview" class="csv-preview" style="display: none;"></div>';
        echo '<p style="margin: 10px 0 0 0; font-size: 0.9em; color: #666;"><strong>PRO:</strong> ' . 
             esc_html__('Taille max 20MB | Auto-détection séparateurs | Encodages multiples', 'easy-alt-import-pro') . '</p>';
        echo '</div>';
        
        submit_button('🚀 ' . __('Analyser et prévisualiser le fichier CSV', 'easy-alt-import-pro'), 'primary', 'submit', false, [
            'style' => 'font-size: 1.1em; padding: 12px 24px; width: 100%;'
        ]);
        echo '</form>';
        
        echo '</div>';
    }

    echo '</div>'; // easy-alt-container

    echo '<div class="easy-alt-footer">';
    echo '<p>🚀 ' . sprintf(
        // translators: %1$s is the heart emoji, %2$s is the company link, %3$s is the version number
        esc_html__('Plugin PRO développé avec %1$s par %2$s | Version %3$s', 'easy-alt-import-pro'), 
        '❤️', 
        '<a href="' . esc_url('https://ftmpub.ch') . '" target="_blank" style="color: #667eea; text-decoration: none;">ftmpub.ch</a>', 
        esc_html(EASY_ALT_IMPORT_VERSION)
    ) . '</p>';
    echo '<p style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #eee; font-size: 0.85em; color: #888;">';
    echo '<strong>' . esc_html__('Support PRO :', 'easy-alt-import-pro') . '</strong> ';
    echo '<a href="https://ftmpub.ch/support-pro" target="_blank" style="color: #007cba;">📧 ' . 
         esc_html__('Contact support', 'easy-alt-import-pro') . '</a> | ';
    echo '<a href="https://ftmpub.ch/easy-alt-import-docs" target="_blank" style="color: #007cba;">📚 ' . 
         esc_html__('Documentation', 'easy-alt-import-pro') . '</a> | ';
    echo '<a href="https://ftmpub.ch/easy-alt-import-changelog" target="_blank" style="color: #007cba;">🔄 ' . 
         esc_html__('Nouveautés', 'easy-alt-import-pro') . '</a>';
    echo '</p>';
    echo '</div>';

    echo '</div>'; // wrap

    // JavaScript pour les fonctionnalités interactives
    echo '<script>
    function selectAll() {
        const checkboxes = document.querySelectorAll(\'input[name="selected_ids[]"]\');
        checkboxes.forEach(cb => cb.checked = true);
        updateCount();
    }
    
    function deselectAll() {
        const checkboxes = document.querySelectorAll(\'input[name="selected_ids[]"]\');
        checkboxes.forEach(cb => cb.checked = false);
        updateCount();
    }
    
    function updateCount() {
        const checked = document.querySelectorAll(\'input[name="selected_ids[]"]:checked\').length;
        const countElement = document.getElementById(\'selected-count\');
        const statElement = document.getElementById(\'selected-stat\');
        if (countElement) countElement.textContent = checked;
        if (statElement) statElement.textContent = checked;
    }
    
    function confirmImport() {
        const checked = document.querySelectorAll(\'input[name="selected_ids[]"]:checked\').length;
        if (checked === 0) {
            alert(\'❌ ' . esc_js(__('Veuillez sélectionner au moins une image à importer.', 'easy-alt-import-pro')) . '\');
            return false;
        }
        return confirm(\'🚀 ' . esc_js(__('Confirmer l\'optimisation SEO de', 'easy-alt-import-pro')) . ' \' + checked + \' ' . 
                      esc_js(__('image(s) ? Cette action va améliorer votre référencement.', 'easy-alt-import-pro')) . '\');
    }
    
    function previewCSV(input) {
        if (input.files[0]) {
            const reader = new FileReader();
            reader.onload = function(e) {
                const lines = e.target.result.split("\\n").slice(0, 4);
                const preview = document.getElementById("csv-preview");
                if (preview && lines.length > 0) {
                    preview.style.display = "block";
                    preview.innerHTML = "<strong>📄 Aperçu des premières lignes :</strong><br>" + 
                        lines.map(line => "<code>" + line.substring(0, 100) + (line.length > 100 ? "..." : "") + "</code>").join("<br>");
                    
                    const firstLine = lines[0] || "";
                    const semicolons = (firstLine.match(/;/g) || []).length;
                    const commas = (firstLine.match(/,/g) || []).length;
                    const tabs = (firstLine.match(/\\t/g) || []).length;
                    
                    let detectedSeparator = "virgules";
                    if (semicolons > commas && semicolons > tabs) detectedSeparator = "point-virgules";
                    else if (tabs > commas && tabs > semicolons) detectedSeparator = "tabulations";
                    
                    preview.innerHTML += "<br><small style=\\"color: #007cba;\\"><strong>🤖 Séparateur détecté :</strong> " + detectedSeparator + " (auto-détection PRO)</small>";
                }
            };
            reader.readAsText(input.files[0]);
        }
    }
    </script>';
}

// Set activation notice on plugin activation
register_activation_hook(__FILE__, function() {
    set_transient('easy_alt_import_pro_activated', true, 30);
    
    // Initialize PRO stats if not exist
    if (!get_option('easy_alt_import_pro_total_stats')) {
        update_option('easy_alt_import_pro_total_stats', [
            'total_processed' => 0,
            'total_imports' => 0,
            'last_import' => null,
            'activation_date' => current_time('mysql')
        ]);
    }
});

/* ========================================
   FIN DE LA PARTIE 4/4 - INTERFACE UTILISATEUR
   ======================================== */

?>
